///
///  This source code is freeware and is provided on an "as is" basis without warranties of any kind, 
///  whether express or implied, including without limitation warranties that the code is free of defect, 
///  fit for a particular purpose or non-infringing.  The entire risk as to the quality and performance of 
///  the code is with the end user.
///

using System;
using System.Collections.Generic;
using System.Text;
using Microsoft.SharePoint;
using Microsoft.SharePoint.Administration;

namespace V3UpdateAlerts
{
    class V3UpdateAlerts
    {
        static void Main(string[] args)
        {
            string strSiteCollection = string.Empty;

            // First (and only) argument is the URL to the Site Collection
            if (args.Length > 0)
            {
                strSiteCollection = args[0];

                try
                {
                    using (SPSite spsitecol = new SPSite(strSiteCollection))
                    {
                        using (SPWeb sprootweb = spsitecol.RootWeb)
                        {
                            ListAndFixAlerts(sprootweb);
                        } // using sprootweb
                    } // using spsitecol
                }
                catch (System.IO.FileNotFoundException ioex)
                {
                    Console.WriteLine("Could not access Site Collection: {0}", ioex.Message);
                }
                catch (Exception ex)
                {
                    Console.WriteLine("An unexpected error occurred: {0}", ex.Message);
                } // try
            }
            else
                Console.WriteLine("No Site Collection URL was specified.");
        }

        //Enumerates all Alerts present on a Site Collection and modifies them
        //so wrong information (e. g. in the SiteUrl column) is corrected
        static void ListAndFixAlerts(SPWeb spweb)
        {
            Console.WriteLine("\n{0} Alert(s) for site: {1} ({2})", spweb.Alerts.Count, spweb.Title, spweb.Url);

            foreach (SPAlert alert in spweb.Alerts)
            {
                try
                {
                    Console.WriteLine(" {2} Alert \"{0}\" for user \"{1}\" \n  (Id: {3})\n", alert.Title, (alert.User != null ? alert.User.Name : "<unknown>"), alert.AlertFrequency.ToString(), alert.ID.ToString());

                    //Preserve the initial Alert frequency (Immed., Daily, Weekly)
                    SPAlertFrequency afPrevious = alert.AlertFrequency;

                    //Change the Alert frequency so the datarow under the hood
                    //is touched by SharePoint API
                    if (afPrevious == SPAlertFrequency.Immediate)
                        alert.AlertFrequency = SPAlertFrequency.Weekly;
                    else
                        alert.AlertFrequency = SPAlertFrequency.Immediate;

                    try
                    {
                        alert.Update();

                        //Change the Alert frequency back to the initial state
                        alert.AlertFrequency = afPrevious;
                        alert.Update();
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(" -> Error changing Alert. {0}", ex.Message);
                    } // inner try
                }
                catch (Exception ex)
                {
                    Console.WriteLine(" An unexpected error occurred while accessing the Alert object: {0}", ex.Message);
                } // outer try
            } // foreach alert

            //Take care of all Sub-Webs recursivly
            if (spweb.Webs.Count > 0)
            {
                for(int i = spweb.Webs.Count -1; i >= 0; i--)
                {
                    using(SPWeb spsubweb = spweb.Webs[i])
                    {
                        ListAndFixAlerts(spsubweb);
                    } // using spsubweb
                } //for subwebs
            } //if has subweb
        }
    }
}
